from typing import Tuple, Dict
import pandas as pd
import numpy as np
from astropy.table import Table

def read_mrt_table(path: str) -> pd.DataFrame:
    tab = Table.read(path, format='ascii.mrt')
    return tab.to_pandas()

def split_mass_models_by_galaxy(df: pd.DataFrame) -> Dict[str, pd.DataFrame]:
    keys = [k for k in df.columns if k.lower() in ['name','galaxy','object','sparcname']]
    if not keys:
        for c in df.columns:
            if df[c].dtype == object and df[c].nunique() < len(df)/2:
                keys = [c]; break
    if not keys:
        raise ValueError("Could not identify galaxy name column")
    k = keys[0]
    groups = {}
    for gname, gdf in df.groupby(k):
        rcols = [c for c in gdf.columns if c.lower().startswith('r')]
        if rcols:
            gdf = gdf.sort_values(by=rcols[0]).reset_index(drop=True)
        groups[str(gname)] = gdf
    return groups

def extract_columns(gdf: pd.DataFrame):
    rcol=None
    for c in gdf.columns:
        lc=c.lower().strip()
        if (lc.startswith('r') and 'kpc' in lc) or lc=='r':
            rcol=c;break
    if rcol is None:
        num=[c for c in gdf.columns if np.issubdtype(gdf[c].dtype, np.number)]
        rcol = num[0]
    R = gdf[rcol].astype(float)

    vobs_col=None
    for c in gdf.columns:
        lc=c.lower()
        if 'vobs' in lc or (lc.startswith('v') and 'obs' in lc):
            vobs_col=c;break
    if vobs_col is None:
        for c in gdf.columns:
            if c.lower() in ['v','vtot','v_tot']:
                vobs_col=c;break
    Vobs = gdf[vobs_col].astype(float)

    comps={}
    for key, patt in [('Vgas','gas'),('Vdisk','disk'),('Vbulge','bul')]:
        col=None
        for c in gdf.columns:
            if (patt in c.lower()) and c!=vobs_col:
                col=c;break
        if col is not None:
            comps[key]=gdf[col].astype(float)
    return R, Vobs, comps

def read_sizes_from_table1(path: str, size_rule: str) -> Dict[str, float]:
    tab = Table.read(path, format='ascii.mrt')
    df = tab.to_pandas()
    name_col=None
    for c in df.columns:
        if c.lower() in ['name','galaxy','sparcname','object']:
            name_col=c;break
    if name_col is None: name_col=df.columns[0]
    rd_col=None; reff_col=None
    for c in df.columns:
        lc=c.lower()
        if ('rd' in lc and 'kpc' in lc) or lc.strip()=='rd': rd_col=c
        if ('reff' in lc) or lc.strip()=='re': reff_col=c
    rg={}
    if size_rule=='RG=2.2*Rd' and rd_col is not None:
        for _,row in df.iterrows():
            try: rg[str(row[name_col])] = 2.2*float(row[rd_col])
            except: pass
    elif size_rule=='RG=Reff' and reff_col is not None:
        for _,row in df.iterrows():
            try: rg[str(row[name_col])] = float(row[reff_col])
            except: pass
    return rg
